//*************************************************************************************************
//
//	Description:
//		temp_grass.fx - Temporary grass shader for the Kart Track. Use this in lots of places
//													and you'll suffer when its eventually deleted!
//
//	<P> Copyright (c) 2007 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     11/07/2007  0.1           Created
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//




//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//
#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
>;

#endif


//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;


//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS

// colour multiplier, forced to end to avoid artists touching it

float4 globalColourMultiplier
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = true;
> = { 1.0f, 1.0f, 1.0f, 1.0f };


//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


//
// Functions
//


//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float3 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord : TEXCOORD0;												// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																																	// as it implicitly uses that to calculate the tangent space coordinate frame.
	float3 alpha		: TEXCOORD2;													// Vertex alpha
#else
	float4 colour   : COLOR0;																// Vertex colour
	float2 texCoord : TEXCOORD0;												// UV channel 1 texture coord
#endif
	float3 normal   : NORMAL;															// Object space normal
};


struct SHADOWGEN_VSINPUT
{
	float3 position : POSITION;														// Object space position
	float4 colour   : COLOR0;																// Vertex colour
	float2 texCoord : TEXCOORD0;												// Texture coords
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 ambient		: TEXCOORD3;														// Ambient light
	float4 colour			: TEXCOORD4;														// Vertex tint
	float2 texCoord	: TEXCOORD0;											// UV coords for texture channel 0
	float3 normal		: TEXCOORD1;													// Normal vector (world space) + fresnel coefficient
	float4 eye				: TEXCOORD2;													// Eye vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 5 )
};



struct SHADOWGEN_VSOUTPUT
{
	float4 position			: POSITION;
	float4 colour				: TEXCOORD2;
	float2 texCoord			: TEXCOORD0;
	float4 shadowCoord	: TEXCOORD1;
};

struct ZPRIMEDOF_VSOUTPUT
{
	float4 position			: POSITION;
	float4 colour				: TEXCOORD2;
	float2 texCoord			: TEXCOORD0;
	float4 coords			: TEXCOORD1;
};


//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT TempGrassVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	_output.normal = normal;

	// Do lighting
#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
	float globalSpecularFactorValue = 0.0f;
	float minSpecPowerValue = 1.0f;
	float maxSpecPowerValue = 1.0f;
#endif

	_output.colour = float4( 1.0f, 1.0f, 1.0f, 1.0f );
	DO_VS_LIGHTING_CALCULATIONS

	// Swap the lighting over to the ambient output register
	_output.ambient = _output.colour;
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	_output.colour *= globalColourMultiplier;

	return _output;
}



SHADOWGEN_VSOUTPUT TempGrassShadowGenVertexShader( SHADOWGEN_VSINPUT _input )
{
	SHADOWGEN_VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord;
	_output.colour	 = _input.colour;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Clamp geometry that is behind the camera to the near plane so that it still renders.
	// We use an orthogonal projection, so doing this will not distort the shadow caster.
	CLAMP_SHADOW_Z;

	OUTPUT_SHADOW_COORDS;

	return _output;
}


ZPRIMEDOF_VSOUTPUT TempGrassZPrimeDOFVertexShader( SHADOWGEN_VSINPUT _input )
{
	ZPRIMEDOF_VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord;
	_output.colour	 = _input.colour;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );
	_output.coords = _output.position;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

struct PSINPUT
{
	float4 ambient			: TEXCOORD3;								// Ambient lighting
	float4 colour			: TEXCOORD4;								// Vertex colour
	float2 texCoord		: TEXCOORD0;						// UV coords for texture channel 0
	float3 normal	: TEXCOORD1;		// Normal vector (world space)
	float4 eye				: TEXCOORD2;						// Eye vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 5 )
};

#else

// Input structure
struct PSINPUT
{
	float4 ambient			: TEXCOORD3;								// Ambient lighting
	float4 colour			: TEXCOORD4;											// Vertex colour
	float2 texCoord		: TEXCOORD0;									// UV coords for texture channel 0
	float3 normal	: TEXCOORD1_centroid;	// Normal vector (world space)
	float4 eye				: TEXCOORD2_centroid;					// Eye vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 5 )
	DECLARE_SHADOW_PS_INPUTS
};

#endif	// defined( _3DSMAX_ )


struct SHADOWGEN_PSINPUT
{
	float4 colour				: TEXCOORD2;
	float2 texCoord			: TEXCOORD0;
	float4 shadowCoord	: TEXCOORD1;
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT TempGrassFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Read texture & factor in tint from vertex data
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord ) * _input.colour;

	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
	float4 specularTexColour = float4( 1.0f, 1.0f, 1.0f, 0.0f );
	float globalSpecularFactorValue = 0.0f;
	float minSpecPowerValue = 0.0f;
	float maxSpecPowerValue = 0.0f;

  // Normalise interpolated vectors
	float3 normal = normalize( _input.normal );
  float3 eye = normalize( _input.eye.xyz );

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.ambient;

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	accumulatedColour.w = diffuseTexColour.w;
	_output.Colour = CalculateOutputPixel( accumulatedColour );

	return _output;
}


PSOUTPUT TempGrassShadowGenFragmentShader( SHADOWGEN_PSINPUT _input )
{
#if !defined(NEEDS_SHADOW_COORDS)
	PSOUTPUT output;
	
	output.Colour=0.0f;
	
	return output;
#else
	PSOUTPUT output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	CALC_SHADOWMAP_DEPTH( output.Colour, _input.shadowCoord );
	output.Colour.a = diffuseTexColour.a;
	SHADOWMAP_PS_ALPHATEST( output.Colour.a, (100.0f/255.0f) );

	return output;
#endif	
}


PSOUTPUT TempGrassZPrimeDOFFragmentShader( ZPRIMEDOF_VSOUTPUT _input )
{
	PSOUTPUT output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	output.Colour = _input.coords.z / _input.coords.w;
	output.Colour.a = diffuseTexColour.a * _input.colour.a;

	return output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique TempGrass
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "TempGrass";
	int    normalDeferredID			= 0;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_RENDER";
	string zprimeDOFTechnique		= "_TempGrass_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_TempGrass_ShadowGen";
	int    shadowGenDeferredID	= 0;
	string lowDetailBehaviour		= "ERMB_DONT_RENDER";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
		bool	AlphaTestEnable = true;
		int		AlphaRef = 100;
		string AlphaFunc = "GreaterEqual";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = true;
		AlphaTestEnable = true;
		AlphaRef = 100;
		AlphaFunc = GreaterEqual;
#endif

#if defined (_PS3_)
			VertexShader = compile sce_vp_rsx TempGrassVertexShader();
			PixelShader = compile sce_fp_rsx TempGrassFragmentShader();
#else
		VertexShader = compile vs_3_0 TempGrassVertexShader();
		PixelShader = compile ps_3_0 TempGrassFragmentShader();
#endif
	}
}


technique _TempGrass_ShadowGen
{
	pass Pass0
	{
			SHADOWMAP_STATES_ALPHATEST( 100 )
#if defined (_PS3_)
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx TempGrassShadowGenVertexShader();
			PixelShader = compile sce_fp_rsx TempGrassShadowGenFragmentShader();
#elif defined (_XBOX)
			VertexShader = compile vs_3_0 TempGrassShadowGenVertexShader();
			PixelShader = compile ps_3_0 TempGrassShadowGenFragmentShader();
#else
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 TempGrassShadowGenVertexShader();
			PixelShader = compile ps_3_0 TempGrassShadowGenFragmentShader();
#endif
	}
}

technique _TempGrass_ZPrime_DOF
{
	pass Pass0
	{
#if defined (_PS3_)
			AlphaTestEnable = true;
		  AlphaFunc = int2(GEqual, 100);
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx TempGrassZPrimeDOFVertexShader();
			PixelShader = compile sce_fp_rsx TempGrassZPrimeDOFFragmentShader();
#else
			AlphaTestEnable = true;
			AlphaRef = 100;
	    AlphaFunc = GreaterEqual;
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 TempGrassZPrimeDOFVertexShader();
			PixelShader = compile ps_3_0 TempGrassZPrimeDOFFragmentShader();
#endif
	}
}
